## Apply tax and super rules to income projections, accounting for assumptions about how tax threshold changes and 
## expected super contribution changes


# Preliminaries -----------------------------------------------------------

rm(list=ls())
gc()



# Read income projection data ---------------------------------------------

income_projection <- qread("./Input data/income_projections_ai.qs") %>% 
  ## assume all income is taxable
  mutate(income = wages_av_smooth + other_inc_av_smooth)

## Get AWE projections for indexing tax and super thresholds
awe_projection <- qread("./Input data/AWE_projections.qs") %>% 
  filter(year>=2022) %>% 
  mutate(awe_growth_factor_2026 = awe_growth_factor_2018 / awe_growth_factor_2018[5],
         awe_growth_factor_2022 = awe_growth_factor_2018 / awe_growth_factor_2018[1]) 



# Tax rates and thresholds ----------------------------------------------------------

## historical tax rates https://www.ato.gov.au/Rates/Individual-income-tax-for-prior-years/
## current tax rates https://www.ato.gov.au/rates/individual-income-tax-rates/
## future tax rates and offets  https://www.ato.gov.au/General/New-legislation/In-detail/Direct-taxes/Income-tax-for-individuals/JobMaker-Plan---bringing-forward-the-Personal-Income-Tax-Plan/

tax_rate_2018 <- data.frame(
  year = 2018, ##2017-18 financial year
  tax_threshold = c(18200, 37000, 87000, 180000), ## interpretation: any value from 18200-37000 is taxed at 19%
  tax_rate = c(0.19, 0.325, 0.37, 0.45)
)

tax_rate_2019 <- data.frame(
  year = 2019, ##2018-19
  tax_threshold = c(18200, 37000, 90000, 180000), 
  tax_rate = c(0.19, 0.325, 0.37, 0.45)
)

tax_rate_2021 <- data.frame(
  year = 2021, ##2020-21
  tax_threshold = c(18200, 45000, 120000, 180000), 
  tax_rate = c(0.19, 0.325, 0.37, 0.45)
)

tax_rate_2025 <- data.frame(
  year = 2025, ##2024-25
  tax_threshold = c(18200, 45000, 200000, 200500), # added a dummy threshold so that all years have 4 thresholds and rates
  tax_rate = c(0.19, 0.30, 0.45, 0.45)
)



## Assume that income tax thresholds are indexed to wage growth from 2026-27 onwards
## This is broadly consistent with assumptions in the RiR pp. 511, 514, 541
## https://treasury.gov.au/sites/default/files/2021-02/p2020-100554-ud06_appendices.pdf


## grow tax tresholds by AWE
tax_rate_2027plus <- lapply(2027:2050,
                            function(x) {
                              tax_rate_2025 %>% 
                                mutate(year = x,
                                       tax_threshold = tax_threshold * awe_projection[year==x, awe_growth_factor_2026]
                                )
                            }) %>% 
  setNames(2027:2050)

## put all years of tax rates together
tax_rates_list <- list(
  `2018` = tax_rate_2018,
  `2019` = tax_rate_2019,
  `2020` = tax_rate_2019, ## same as 2019
  `2021` = tax_rate_2021,
  `2022` = tax_rate_2021,
  `2023` = tax_rate_2021,
  `2024` = tax_rate_2021, ## unchanged from 2021
  `2025` = tax_rate_2025,
  `2026` = tax_rate_2025 ## assume 2026 is unchanged
) %>% 
  append(tax_rate_2027plus)


## incorporate medicare levy - simplified version based on individual rates without tapering. ignore the medicare levy surcharge
medicare_levy <- 0.02

## assume if your income is above a threshold (growing at ~2.5% CPI), you pay medicare levy of 2%
## future growth with Budget forecasts of CPI
## https://www.ato.gov.au/Individuals/Medicare-and-private-health-insurance/Medicare-levy/Medicare-levy-reduction-for-low-income-earners/
## https://www.ato.gov.au/Individuals/Tax-return/2021/Tax-return/Medicare-levy-questions-M1-M2/M1-Medicare-levy-reduction-or-exemption-2021/#:~:text=Completing%20your%20tax%20return%20%E2%80%93%20Medicare%20levy%20reduction&text=If%20you%20had%20a%20spouse,details%20on%20your%20tax%20return.

# https://iknow.cch.com.au/document/xatagracomUio3008819sl978757464/key-rate-2017-18-medicare-levy-and-low-income-thresholds
# https://iknow.cch.com.au/document/xatagracomUio3094077sl1037744678/key-rate-2018-19-medicare-levy-and-low-income-thresholds

medicare_levy_thresholds <- data.frame(
  year = c(2018:2050),
  ## known thresholds to 2021
  threshold = c(27476, 27998, 28501, rep(29032, 2050-2020))
) %>% 
   ## threshold grows with CPI
  mutate(threshold = case_when(
    year==2022 ~ threshold*1.0175, ## from Budget p.37 https://budget.gov.au/2021-22/content/bp1/download/bp1_bs2.pdf
    year==2023 ~ threshold*1.0175*1.0225, ## Budget
    year>2023 ~ threshold*1.0175*1.0225*1.025^(year-2023), ## long run 2.5%
    TRUE ~ threshold
  )) %>% 
  as.data.table



# Apply tax rules to income projections ---------------------------------------------------------

## calculate income tax
## for each income projection...
income_tax_vector_fn <- function(income_projection) {
  sapply(1:nrow(income_projection) , 
         function(x) {
           ## get the year and income
           year <- income_projection$year[x] %>% as.character
           income <- income_projection$income[x]
           
           ## calc the tax - implemented in a similar way to DAE model
           ## interpreting the first line of code below: get the lowest tax rate (19% in 2018). Calculate income minus first tax threshold - if it's negative, then the tax rate is multiplied by 0. 
           ##                                      if it's positive, then is it higher than the second tax threshold? if not, then the tax rate is multiplied by the difference between the income and first tax threshold.
           ##                                      if income is higher than second tax threshold, the tax rate is multiplied by the diff between second and first tax threshold, and the rest of the tax is calc'd in subsequent steps
           income_tax <- tax_rates_list[[year]]$tax_rate[1] * max( 0, min( income - tax_rates_list[[year]]$tax_threshold[1] , tax_rates_list[[year]]$tax_threshold[2] - tax_rates_list[[year]]$tax_threshold[1] ) ) +
             tax_rates_list[[year]]$tax_rate[2] * max( 0, min(income-tax_rates_list[[year]]$tax_threshold[2] , tax_rates_list[[year]]$tax_threshold[3] - tax_rates_list[[year]]$tax_threshold[2] ) ) +
             tax_rates_list[[year]]$tax_rate[3] * max( 0, min(income-tax_rates_list[[year]]$tax_threshold[3] , tax_rates_list[[year]]$tax_threshold[4] - tax_rates_list[[year]]$tax_threshold[3] ) ) +
             tax_rates_list[[year]]$tax_rate[4] * max( 0, income-tax_rates_list[[year]]$tax_threshold[4]  ) 
         }
  )
}
income_tax_vector <- income_tax_vector_fn(income_projection)

## marginal tax rate - for determining how much returns from "other assets" should be taxed by in the model
marginal_rate_vector_fn <- function(income_projection) {
  sapply(1:nrow(income_projection) , 
         function(x) {
           ## get the year and income
           year <- income_projection$year[x] %>% as.character
           income <- income_projection$income[x]
           
           marginal_rate <- case_when(
             income < tax_rates_list[[year]]$tax_threshold[1] ~ 0,
             income < tax_rates_list[[year]]$tax_threshold[2] ~ tax_rates_list[[year]]$tax_rate[1],
             income < tax_rates_list[[year]]$tax_threshold[3] ~ tax_rates_list[[year]]$tax_rate[2],
             income < tax_rates_list[[year]]$tax_threshold[4] ~ tax_rates_list[[year]]$tax_rate[3],
             income >= tax_rates_list[[year]]$tax_threshold[4] ~ tax_rates_list[[year]]$tax_rate[4]
           )
         }
  )
}
marginal_rate_vector <- marginal_rate_vector_fn(income_projection)


## calculate medicare levy
medicare_levy_vector_fn <- function(income_projection) { 
  sapply(1:nrow(income_projection) , 
         function(x) {
           ## get the year and income
           yearx <- income_projection$year[x]
           income <- income_projection$income[x]
           
           ## calc the medicare levy
           medicare_levy <- ifelse(income > medicare_levy_thresholds[medicare_levy_thresholds$year==yearx, threshold],
                                   medicare_levy*income,
                                   0
           )
         }
  )
}
medicare_levy_vector <- medicare_levy_vector_fn(income_projection)


# Superannuation contribution rates ---------------------------------------

## super guarantee percentages https://www.ato.gov.au/rates/key-superannuation-rates-and-thresholds/?anchor=Superguaranteepercentage

super_rates <- data.frame(
  year=c(2018:2050)
) %>% 
  mutate(
    super_rate = case_when(
      year<=2021 ~ 0.095,
      year==2022 ~ 0.10,
      year==2023 ~ 0.105,
      year==2024 ~ 0.11,
      year==2025 ~ 0.115,
      year>=2026 ~ 0.12
    )
  ) %>% 
  data.table

super_tax_rate <- 0.15 


# Apply super rules to income projections ---------------------------------

super_contrib_vector_fn <- function(income_projection) {
  sapply(1:nrow(income_projection) , 
         function(x) {
           ## get the year and income
           yearx <- income_projection$year[x]
           wages <- income_projection$wages_av_smooth[x]
           
           ## calc the net super contribution from wages
           super_contrib <- wages * super_rates[super_rates$year==yearx, super_rate] * (1-super_tax_rate)
         }
  )
}
super_contrib_vector <- super_contrib_vector_fn(income_projection)



# Put income, tax, super projections together and save --------------------

income_tax_super_projection <- income_projection %>% 
  cbind(income_tax = income_tax_vector,
        medicare_levy = medicare_levy_vector,
        marginal_tax = marginal_rate_vector,
        super_contrib = super_contrib_vector
        )

qsave(income_tax_super_projection, "./Input data/income_tax_super_projection_ai.qs")




